//package globalExamples;

import oracle.express.olapi.data.full.ExpressDataProvider;
import oracle.olapi.metadata.mdm.*;
import oracle.olapi.data.source.Source;

import java.util.List;
import java.util.Iterator;

/**
 * Discovers the MDM metadata objects in the Global schema in the
 * Oracle OLAP Catalog.
 *
 * This is Example 4-9, Discovering the OLAP Catalog Metadata, from
 * Chapter 4, Discovering the Available Metadata, in the Oracle OLAP
 * Developer's Guide to the OLAP API.
 *
 * This program uses the MyConnection10g class.
 *
 * @author Oracle Corporation
 */
public class SampleMetadataDiscoverer10g
{
/**
 * Constant to use to display less information about metadata objects.
 */
static final int TERSE = 0;

/**
 * Constant to use to display more information about metadata objects.
 */
static final int VERBOSE = 1;

private MdmSchema root = null;
private MdmPrimaryDimension mdmDim = null;

  public SampleMetadataDiscoverer10g()
  {
  }

  /**
   * Creates an object that makes a connection to an Oracle database,
   * and gets MDM metadata objects.
   */
  public void run(String[] args)
  {
    // Connect through JDBC to an instance of an Oracle database
    // and get a DataProvider.
    MyConnection10g myConn = new MyConnection10g();
    ExpressDataProvider dp = myConn.connectToDB(args, TERSE);
    //ExpressDataProvider dp = myConn.connectToDB(args, VERBOSE);

    // Get the default MdmMetadataProvider from the DataProvider
    MdmMetadataProvider mp = null;
    try
    {
      mp = (MdmMetadataProvider) dp.getDefaultMetadataProvider();
    }
    catch (Exception e)
    {
      System.out.println("Cannot create the MDM metadata provider. " + e);
    }

    // Get metadata information about the root MdmSchema and its subschemas
    try
    {
      root = mp.getRootSchema();
      System.out.println("The root MdmSchema is " + root.getName() + ".\n");
      getSchemaInfo(root, VERBOSE);
    }
    catch (Exception e)
    {
      System.out.println("Encountered exception " + e);
    }

    // Get the Source for the dimension that was saved in getDimInfo.
    System.out.println("\nGetting the primary Source object for dimension " +
                          mdmDim.getName() + ".");
    Source dimSource = mdmDim.getSource();
    System.out.println("Got the Source.");

    // Close the ExpressDataProvider and the connection
    dp.close();
    System.out.println("\nClosed the DataProvider.");
    myConn.closeConnection();
    System.out.println("Closed the connection.");
  }

  /**
   * Gets information about an MdmSchema.
   */
  public void getSchemaInfo(MdmSchema schema, int outputStyle)
  {
    if (schema == root)
    {
      System.out.println("The MdmPrimaryDimension components of" +
                         " the root schema are:");
    }
    else
    {
      System.out.println("Schema: " + schema.getName());
      System.out.println("   The MdmPrimaryDimension components of schema "
                          + schema.getName() + " are:");
    }
    // Get the dimension information for the MdmSchema
    MdmPrimaryDimension oneDim = null;
    int i = 1;
    try
    {
      List dims = schema.getDimensions();
      Iterator dimIter = dims.iterator();
      // Save the first dimension to use later for getting its Source.
      mdmDim = (MdmPrimaryDimension) dims.get(0);
      // Iterate through the list of MdmPrimaryDimension objects and get
      // information about each one.
      while (dimIter.hasNext())
      {
        oneDim = (MdmPrimaryDimension) dimIter.next();
        getDimInfo(i, oneDim, outputStyle);
        i++;
      }
    }
    catch (Exception e)
    {
      System.out.println("   Encountered exception " + e);
    }

    // If the MdmSchema is the root MdmSchema, get the
    // MdmMeasureDimension amd get its measures.
    MdmMeasure oneMeasure = null;

    MdmMeasureDimension mdmMeasureDim =
                         (MdmMeasureDimension) schema.getMeasureDimension();
    if (mdmMeasureDim != null)
    {
      System.out.println("The measures of the MdmMeasureDimension are:");
      List mdmMeasures = mdmMeasureDim.getMeasures();
      Iterator mdmMeasuresIter = mdmMeasures.iterator();
      while (mdmMeasuresIter.hasNext())
      {
        oneMeasure = (MdmMeasure) mdmMeasuresIter.next();
        getMeasureInfo(oneMeasure, outputStyle);
        System.out.println(" ");
      }
    }

    // Get the measures from the MdmSchema
    try
    {
      List mdmMeasures = schema.getMeasures();
      if (mdmMeasures.size() > 0) {
        Iterator mdmMeasuresIter = mdmMeasures.iterator();
        System.out.println("\n   The measures of schema " +
                           schema.getName() + " are:");
        while (mdmMeasuresIter.hasNext())
        {
          oneMeasure = (MdmMeasure) mdmMeasuresIter.next();
          getMeasureInfo(oneMeasure, outputStyle);
        }
      }
    }
    catch (Exception e)
    {
        System.out.println("   Encountered exception " + e);
    }

    // Get the subschema information for the MdmSchema
    MdmSchema oneSchema = null;
    try
    {
      List subSchemas = schema.getSubSchemas();
      Iterator subSchemaIter = subSchemas.iterator();

      while (subSchemaIter.hasNext())
      {
        oneSchema = (MdmSchema) subSchemaIter.next();
        // To get information on subschemas other than the Global
        // schema, GLOBAL_CAT, remove the if condition and call
        // the getSchemaInfo method.
        //if (oneSchema.getName().equals("GLOBAL_CAT"))
          getSchemaInfo(oneSchema, TERSE);
      }
    }
    catch (Exception e)
    {
        System.out.println("   Encountered exception " + e);
    }
  }

  /**
   * Gets information about an MdmMeasure.
   */
  public void getMeasureInfo(MdmMeasure measure, int outputStyle)
  {
    System.out.println("   Measure: " + measure.getName());
    if (outputStyle == VERBOSE) {
      // Get the dimensions of the MdmMeasure
      try
      {
        List mDims = measure.getDimensions();
        Iterator mDimIter = mDims.iterator();
        System.out.println("   Its dimensions are: ");
        while (mDimIter.hasNext())
        {
          System.out.println("   " +
                             ((MdmDimension) mDimIter.next()).getName());
        }
      }
      catch (Exception e)
      {
        System.out.println("   Encountered exception " + e);
      }
    }
  }

  /**
   * Gets information about an MdmDimension.
   */
  public void getDimInfo(int count,
                         MdmPrimaryDimension dim,
                         int outputStyle)
  {
    if (outputStyle == TERSE)
      System.out.println("   " + dim.getName());

    else if (outputStyle == VERBOSE)
    {
      System.out.println(count + ": MdmPrimaryDimension Name: " +
                         dim.getName());
      String description = dim.getDescription();
      if (description.length() > 0)
        System.out.println("   Description: " + dim.getDescription());
      // Determine the type of the MdmPrimaryDimension
      try
      {
        if (dim instanceof MdmStandardDimension)
        {
          System.out.println("   It is an MdmStandardDimension.");
        }
        else if (dim instanceof MdmTimeDimension)
        {
          System.out.println("   It is an an MdmTimeDimension.");
        }
        else if (dim instanceof MdmMeasureDimension)
        {
          System.out.println("   It is an MdmMeasureDimension.");
        }
      }
      catch (Exception e)
      {
          System.out.println("   Encountered exception " + e);
      }

      // Get the attributes of the MdmPrimaryDimension
      System.out.println("   Its attributes are:");
      try
      {
        List attributes = dim.getAttributes();
        Iterator attrIter = attributes.iterator();
        while (attrIter.hasNext())
        {
          System.out.println("   Attribute: " +
                            ((MdmAttribute) attrIter.next()).getName());
        }
      }
      catch (Exception e)
      {
        System.out.println("   Encountered exception " + e);
      }

      // Get the hierarchies of the MdmPrimaryDimension
      getHierInfo(dim);
      System.out.println(" ");
    }
  }

  /**
   * Gets the MdmHierarchy components of an MdmPrimaryDimension.
   */
  public void getHierInfo(MdmPrimaryDimension dim)
  {
    List mdmHiers = dim.getHierarchies();
    Iterator mdmHiersItr = mdmHiers.iterator();
    MdmHierarchy mdmHier = null;
    MdmLevelHierarchy mdmLevelHier = null;
    boolean isLevelHierarchy = false;
    int i = 1;
    System.out.println("   The MdmHierarchy components of " +
                        dim.getName() + " are:");
    while (mdmHiersItr.hasNext())
    {
      mdmHier = (MdmHierarchy) mdmHiersItr.next();
      System.out.println("   "+ i + ": " + mdmHier.getName());
      if (mdmHier.isDefaultHierarchy())
      {
        System.out.println("      " + mdmHier.getName() + " is the default" +
                           " MdmHierarchy of " + dim.getName() + ".");
      }
      if (mdmHier instanceof MdmLevelHierarchy)
      {
        mdmLevelHier = (MdmLevelHierarchy) mdmHier;
        System.out.println("      It is an MdmLevelHierarchy.");
        isLevelHierarchy = true;
      }
      else if (mdmHier instanceof MdmValueHierarchy)
      {
        System.out.println("      It is an MdmValueHierarchy");
      }
      System.out.println("      Its attributes are:");
      if (isLevelHierarchy)
      {
        System.out.println("      Level attribute: "
                          + mdmLevelHier.getLevelAttribute().getName());
        System.out.println("      Level depth attribute: "
                           + mdmLevelHier.getLevelDepthAttribute().getName());
      }
      System.out.println("      Parent attribute: " +
                          mdmHier.getParentAttribute().getName());
      System.out.println("      Ancestors attribute: " +
                          mdmHier.getAncestorsAttribute().getName());
      if (isLevelHierarchy)
        getLevelInfo(mdmLevelHier);
     i++;
    }
  }

  /**
   * Gets the MdmLevel components of an MdmLevelHierarchy.
   */
  public void getLevelInfo(MdmLevelHierarchy mdmHier)
  {
    List mdmLevels = mdmHier.getLevels();
    Iterator mdmLevelsItr = mdmLevels.iterator();
    System.out.println("      Its levels are:");
    while (mdmLevelsItr.hasNext()) {
      MdmLevel mdmLevel = (MdmLevel) mdmLevelsItr.next();
      System.out.println("      " + mdmLevel.getName());
    }
  }

  /**
   * Creates a new SampleMetadataDiscoverer10g object and calls its
   * run method.
   *
   * @param args The command line arguments supplied to the Java compiler.
   */
  public static void main(String[] args)
  {
    new SampleMetadataDiscoverer10g().run(args);
  }
}
